
import numpy as np

# Вычисление параметров Хилласа для одного изображения

def HP_calc(xx, yy, ampl, src_x, src_y):
	"""
	Входные данные:
	- xx и уу - одномерные массивы координат пикселей изображения
	- ampl - одномерный массив изображения
	- src_x и src_y - центр системы координат камеры, в которой необходимо рассчитать параметры
	
	Выходные данные - вектор параметров Хилласа. Их порядок:
	- 0 - чилсо ненулевых пикселей в изображении
	- 1 - х-координата CoG эллипса
	- 2 - y-координата CoG эллипса
	- 3 - distance
	- 4 - length
	- 5 - width
	- 6 - azwidth
	- 7 - miss
	- 8 - угол альфа
	- 9 - угол фи (Угол между OX-осью и Length)
	- 10 - угол тета (полярный угол)
	- 11 - концентрация
	- 12 - десятичный логарифм size
	- 13 - асимметрия вдоль большой полуоси эллипса
	- 14 - асимметрия вдоль малой полуоси эллипса
	- 15 - эксцесс вдоль большой полуоси эллипса
	- 16 - эксцесс вдоль малой полуоси эллипса
	Примечание:
	- Размерность пространственных параметров (например, х_CoG, distance, length и т.д.) соответствует размерности xx и yy-массивов
	- Размерность угловых параметров (альфа, фи и тета) выражеы в радианах
	"""
	
	if np.all(ampl == 0.):
		return([0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., np.log10(1.), 0., 0., 0., 0., 0.])
	
	xx = xx - src_x
	yy = yy - src_y
	
	x_mean = np.average(xx, weights=ampl)
	y_mean = np.average(yy, weights=ampl)
	x2_mean = np.average(xx**2, weights=ampl)
	y2_mean = np.average(yy**2, weights=ampl)
	xy_mean = np.average(xx*yy, weights=ampl)
	sig_xx = x2_mean - x_mean**2
	sig_yy = y2_mean - y_mean**2
	sig_xy = xy_mean - x_mean*y_mean
	if (sig_xx<0.): sig_xx = 0.
	if (sig_yy<0.): sig_yy = 0.
	z = np.sqrt( (sig_yy - sig_xx)**2 + 4.*sig_xy**2 )
	
	N_pix = np.count_nonzero(ampl)
	size = np.sum(ampl[ampl>0.000001])
	dist = np.sqrt(x_mean**2 + y_mean**2)
	length = np.sqrt((sig_xx + sig_yy + z)/2.)
	width  = (sig_xx + sig_yy - z)/2.
	if (width<0.): width = 0.
	if (dist==0.): dist = 0.001
	width = np.sqrt(width)
	conc  = np.sum(-np.partition(-ampl, 2)[:2])/size
	if sig_xy !=0.:
		a = ((sig_yy - sig_xx)+z) / (2.*sig_xy)
		miss   =  np.abs(y_mean - a*x_mean)/np.sqrt(1+a**2)
	else: 
		a=0.
		miss = 0.
	alpha  = np.arcsin(miss/dist)
	
	# Угол между центром тяжести и центром координат - Teta
	sinTet = y_mean/dist;
	cosTet = x_mean/dist;
	if (cosTet>=0)and(sinTet>=0) : Teta = np.arcsin(sinTet)
	elif (cosTet<0)and(sinTet>=0): Teta = np.pi - np.arcsin(sinTet)
	elif (cosTet<0)and(sinTet<0) : Teta = np.pi + np.abs( np.arcsin(sinTet) )
	elif (cosTet>=0)and(sinTet<0): Teta = 2*np.pi + np.arcsin(sinTet) 
	# Угол между OX-осью и Length - Phi
	VyPhi = 2.*sig_xy*x_mean + (z-(sig_xx-sig_yy))*y_mean
	VxPhi = (z+(sig_xx-sig_yy))*x_mean + 2.*sig_xy*y_mean
	cosPhi = VxPhi/np.sqrt(VxPhi**2+VyPhi**2)
	sinPhi = VyPhi/np.sqrt(VxPhi**2+VyPhi**2)
	if (VxPhi == 0):
		VxPhi = 1.; VyPhi = 0.
	if (VxPhi>=0)and(VyPhi>=0): Phi = np.arctan( VyPhi/VxPhi )
	elif ((VxPhi<0)and(VyPhi>=0)) or ((VxPhi<0)and(VyPhi<0)): Phi = np.arctan( VyPhi/VxPhi ) + np.pi
	elif (VxPhi>=0)and(VyPhi<0): Phi = np.arctan( VyPhi/VxPhi ) + 2*np.pi

	# Частичное преобразование координат
	xx_phi = np.cos(Phi)*(xx-x_mean) + np.sin(Phi)*(yy-y_mean)
	yy_phi = -np.sin(Phi)*(xx-x_mean) + np.cos(Phi)*(yy-y_mean)
	#xx_tet =  cosTet*(xx-x_mean) + sinTet*(yy-y_mean)
	yy_tet = -sinTet*(xx-x_mean) + cosTet*(yy-y_mean)
	x_phi_mean  = np.average(xx_phi, weights=ampl)
	x2_phi_mean = np.average((xx_phi-x_phi_mean)**2, weights=ampl)
	x3_phi_mean = np.average((xx_phi-x_phi_mean)**3, weights=ampl)
	x4_phi_mean = np.average((xx_phi-x_phi_mean)**4, weights=ampl)
	y_phi_mean  = np.average(yy_phi, weights=ampl)
	y2_phi_mean = np.average((yy_phi-y_phi_mean)**2, weights=ampl)
	y3_phi_mean = np.average((yy_phi-y_phi_mean)**3, weights=ampl)
	y4_phi_mean = np.average((yy_phi-y_phi_mean)**4, weights=ampl)
	
	azwidth = np.average(yy_tet**2, weights=ampl) -(np.average(yy_tet, weights=ampl))**2
	if azwidth <0.: azwidth = 0.
	else: azwidth = np.sqrt(azwidth)
	asym_L   = x3_phi_mean / np.sqrt(x2_phi_mean**3)	
	kurt_L   = x4_phi_mean / np.sqrt(x2_phi_mean**4) -3.
	
	if (np.sqrt(y2_phi_mean) <0.000001):
		asym_W = 0.
		kurt_W = -3.
	else:
		asym_W   = y3_phi_mean / np.power(np.sqrt(y2_phi_mean),3)
		kurt_W   = y4_phi_mean / np.power(np.sqrt(y2_phi_mean),4) - 3.
	
	return([N_pix, x_mean, y_mean, dist, length, width, azwidth, miss, alpha, Phi, Teta, conc, np.log10(size), asym_L, \
			asym_W, kurt_L, kurt_W])

if __name__ == "__main__":
	## Тестовое изображение
	xx = np.random.choice(np.arange(0,100,1), size=100, replace=False)/10.
	yy = np.random.choice(np.arange(0,100,1), size=100, replace=False)/10.
	print(xx,yy)
	image = np.zeros((100));
	ind = np.arange(20, 60, 1)
	image[ind] = 1.
	
	# Вычисление вектора с параметрами Хилласа для этого изображения
	HP = HP_calc(xx, yy, image, 0., 0.)
	print(HP)
